package {
	import flash.events.EventDispatcher;
	import flash.system.Security;
	import flash.utils.ByteArray;

	import mx.controls.Alert;
	import mx.core.Application;
	import mx.events.SWFBridgeRequest;
	import mx.managers.SystemManager;
	import mx.utils.Base64Decoder;
	import mx.utils.Base64Encoder;

	/**
	 * ContentStationAPI provides methods for SubApplications to access ContentStation functions
	 *
	 * @author     Gaelle Meheut
	 * @since       v7.1
	 * @copyright	Woodwing Software bv. All Rights Reserved.
	 **/

	public class ContentStationAPI {
		// Events   
		public static const CS_API_GET_TICKET:String = "CS_API_GetTicket";
		public static const CS_API_GET_USERNAME:String = "CS_API_GetUsername";
		public static const CS_API_GET_USERID:String = "CS_API_GetUserID";
		public static const CS_API_GET_SERVERURL:String = "CS_API_GetServerUrl";
		public static const CS_API_DEBUG:String = "CS_API_Debug";
		public static const CS_API_SESSION_CLOSING:String = "CS_API_Session_Closing";
		public static const CS_API_CLOSE_SESSION:String = "CS_API_Close_Session";
		public static const CS_API_CANCEL_CLOSE_SESSION:String = "CS_API_Cancel_Close_Session";
		public static const CS_API_TAB_CLOSE_CANCELED:String = "CS_API_Tab_Close_Canceled";
		public static const CLEAR_ALL_LISTENERS:String = "CS_API_Clear_All_Listeners";
		public static const CS_API_GET_BASEPATH:String = "CS_API_GetBasePath";
		public static const CS_API_OPEN_SMARTOBJECT:String = "CS_API_OpenSmartObject";
		public static const CS_API_REFRESH_SMARTOBJECT:String = "CS_API_RefreshSmartObject";
		public static const CS_API_GET_CONFIGSETTING:String = "CS_API_GetConfigSetting";

		//API's added since 8.2
		public static const CS_API_GET_SERVER_FEATURE:String = "CS_API_GetServerFeature";
		public static const CS_API_CREATE_SUB_APP_MENU:String = "CS_API_CreateSubAppMenu";
		public static const CS_API_IS_TAB_SELECTED:String = "CS_API_IsTabSelected";
		public static const CS_API_SAVE_SHARED_OBJECT:String = "CS_API_SaveSharedObject";
		public static const CS_API_GET_SHARED_OBJECT:String = "CS_API_GetSharedObject";

		/**
		 * Event will be dispatched when the tab selected
		 */
		public static const CS_API_TAB_SELECTED_EVENT:String = "CS_API_TabSelectedEvent";
		/**
		 * Event will be dispatched when the tab unselected
		 */
		public static const CS_API_TAB_UNSELECTED_EVENT:String = "CS_API_TabUnselectedEvent";

		public var sharedEvents:EventDispatcher;
		public var handler:Function;

		/**
		 * Constructor
		 * The sharedEvents should be the same for subApplications and ContentStation.
		 * As the subApplications are opened within ContentStation, the 2 applications are sharing the same system Manager and sharedEvents.
		 *
		 * @param	apiDispatcher EventDispatcher :
		 **/
		public function ContentStationAPI (sharedEvents:EventDispatcher, handler:Function = null):void {
			this.sharedEvents = sharedEvents;
			//suscribed to events that can be dispatched by the ContentStationAPI

			if (handler != null) {
				this.handler = handler;
			}

			sharedEvents.addEventListener(CS_API_SESSION_CLOSING, isClosePrevented);
			sharedEvents.addEventListener(CLEAR_ALL_LISTENERS, clearAllListeners);
		}


		/**
		 * launch an event in order to retrieve the ticket of the current ContentStation
		 *
		 * @return ticket String
		 **/
		public function getTicket ():String {
			var requestTicket:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_TICKET);
			this.sharedEvents.dispatchEvent(requestTicket);
			return requestTicket.data.toString();
		}


		/**
		 * launch an event in order to retrieve the username of the current ContentStation
		 *
		 * @return username String
		 **/
		public function getUsername ():String {
			var requestUsername:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_USERNAME);
			sharedEvents.dispatchEvent(requestUsername);
			return requestUsername.data.toString();
		}

		/**
		 *
		 * Added in 8.2, returns the UserID
		 *
		 * launch an event in order to retrieve the userid of the current ContentStation
		 *
		 * @return username String
		 **/
		public function getUserID ():String {
			var requestUserID:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_USERID);
			sharedEvents.dispatchEvent(requestUserID);
			return requestUserID.data.toString();
		}


		/**
		 * launch an event in order to retrieve the server url of the current ContentStation
		 *
		 * @return serverUrl String
		 **/
		public function getServerUrl ():String {
			var requestUrl:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_SERVERURL);
			sharedEvents.dispatchEvent(requestUrl);
			return requestUrl.data.toString();
		}


		/**
		 * check if the close session is prevented by the subapp or module
		 * if an handler is defined, then we ask the handler, if no handler are defined, the close isn't prevented.
		 *
		 * @param event has no Type otherwise it would failed in the case of a Subapplication sandboxed loaded in CS
		 *
		 **/
		private function isClosePrevented (event:Object):void {
			if (this.handler != null) {
				event.data = this.handler();
			}
			else {
				event.data = false;
			}
		}

		/**
		 * Launch an event to be retrieve by the api bridge in order to close the session.
		 *
		 *
		 **/
		public function closeSessionContentStation ():void {
			// needed to prevent multiple call of isClosePrevented by the module after that a session has been closed 
			var requestCloseCS:SWFBridgeRequest = new SWFBridgeRequest(CS_API_CLOSE_SESSION);
			sharedEvents.dispatchEvent(requestCloseCS);
		}


		/**
		 * Launch an event to be retrieve by the api bridge in order to cancel the close of the session.
		 *
		 *
		 **/
		public function cancelCloseSessionContentStation ():void {
			// needed to prevent multiple call of isClosePrevented by the module after that a session has been closed 
			//		sharedEvents.removeEventListener(CS_API_SESSION_CLOSING, isClosePrevented);
			var requestCancelLoggOffCS:SWFBridgeRequest = new SWFBridgeRequest(CS_API_CANCEL_CLOSE_SESSION);
			sharedEvents.dispatchEvent(requestCancelLoggOffCS);
		}

		/**
		 * launch an event in order to display a debug log
		 * level can be : LogEventLevel.FATAL,LogEventLevel.ERROR, LogEventLevel.WARN, LogEventLevel.INFO, LogEventLevel.DEBUG, LogEventLevel.ALL
		 *
		 **/
		public function debug (text:String, level:int = 4):void {
			var requestDebug:SWFBridgeRequest = new SWFBridgeRequest(CS_API_DEBUG, false, false, null, [text, level]);
			sharedEvents.dispatchEvent(requestDebug);
		}

		/**
		 * @param event has no Type otherwise it would failed in the case of a Subapplication sandboxed loaded in CS
		 */
		public function clearAllListeners (event:Object):void {
			sharedEvents.removeEventListener(CS_API_SESSION_CLOSING, isClosePrevented);
			sharedEvents.removeEventListener(CLEAR_ALL_LISTENERS, clearAllListeners);
		}

		/**
		 * Returns the base path of the sub application can be a URI or URL
		 */
		public function getBasePath ():String {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_BASEPATH);
			this.sharedEvents.dispatchEvent(request);
			return request.data.toString();
		}

		/**
		 * Open a SmartObject like article, dossier, InCopy or InDesign document, etc
		 * ArticleTemplates and DossierTemplates will be instantiated
		 * @param  objectID ID of the SmartObject
		 */
		public function openSmartObject (objectID:String):void {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_OPEN_SMARTOBJECT, false, false, null, [objectID]);
			sharedEvents.dispatchEvent(request);
		}

		/**
		 * refresh a SmartObject like article, dossier, InCopy or InDesign document, etc
		 * ArticleTemplates and DossierTemplates will be refreshed
		 * @param  objectID ID of the SmartObject
		 */
		public function refreshSmartObject (objectID:String):void {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_REFRESH_SMARTOBJECT, false, false, null, [objectID]);
			sharedEvents.dispatchEvent(request);
		}

		/**
		 * Returns a setting from the ContentStation area in WWSettings.xml
		 * @param name Name of the setting
		 */
		public function getConfigSetting (name:String):String {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_CONFIGSETTING, false, false, null, [name]);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null) {
				return request.data.toString();
			}
			else {
				return null;
			}
		}

		/////////////////////////
		///  API's since 8.2   //
		/////////////////////////
		/**
		 * Get the value of the requested server feature.
		 *
		 * @param name 	Name of the server feature to get.
		 * @return 	 	The found value of the requested server feature, null if is not.
		 * @since		8.2
		 */
		public function getServerFeature (name:String):String {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_SERVER_FEATURE, false, false, null, [name]);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null) {
				return request.data.toString();
			}
			else {
				return null;
			}
		}

		/**
		 * Create a new native menu with the passed name.
		 * This menu will be unique for all instances of a sub application.
		 *
		 * @param 	name	The name of the menu to create.
		 * @return 	Return the index where the menu has been inserted, -1 if is not.
		 * @since	8.2
		 */
		public function createSubApplicationMenu (name:String):int {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_CREATE_SUB_APP_MENU, false, false, null, [name]);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null)
				return request.data as int;
			else
				return -1;
		}

		/**
		 * Determines whether the tab is selected or not.
		 *
		 * @return	True if the tab is selectd, False if is not.
		 * @since	8.2
		 */
		public function isTabSelected ():Boolean {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_IS_TAB_SELECTED, false, false, null, []);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null)
				return request.data;
			return false;
		}


		/**
		 * Saves the passed object in the sharedObjects with the objectName as key.
		 *
		 * @param subApplicationKey	The unique key used for saving the shared objects of the current sub application.
		 * @param objectName		The name used to save the object
		 * @param object			The object to save.
		 * @return True if the object is saved, False if is not.
		 * @since 8.2
		 */
		public function saveSharedObject (subApplicationKey:String, objectName:String, object:Object):Boolean {
			var bytes:ByteArray = new ByteArray();
			bytes.writeObject(object);
			bytes.position = 0;
			var be:Base64Encoder = new Base64Encoder();
			be.encodeBytes(bytes);

			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_SAVE_SHARED_OBJECT, false, false, null, [subApplicationKey, objectName, be.drain()]);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null)
				return request.data;
			else
				return false;
		}

		/**
		 * Get the saved object from the sharedObjects based on its name.
		 *
		 * @param subApplicationKey	The unique key used for getting the shared objects of the current sub application.
		 * @param objectName		The name used to get the object
		 * @return	Saved Object value if found, null if not.
		 * @since 8.2
		 */
		public function getSharedObject (subApplicationKey:String, objectName:String):Object {
			var request:SWFBridgeRequest = new SWFBridgeRequest(CS_API_GET_SHARED_OBJECT, false, false, null, [subApplicationKey, objectName]);
			this.sharedEvents.dispatchEvent(request);
			if (request.data != null) {
				var dec:Base64Decoder = new Base64Decoder();
				dec.decode(request.data.toString());
				var result:ByteArray = dec.toByteArray();
				result.position = 0;
				return result.readObject();
			}
			else {
				return null;
			}
		}

		/////////////////////////
		///  END API's 8.2   //
		/////////////////////////
	}
}
